#
# file:     parser.py
#
# about:    parsing and unparsing of game messages
#
# author:   Tom "Knio" Flanagan
#
# contact:  theknio@gmail.com
#           irc.freenode.net - Knio - #python #pygame #py
#           * comments welcome *
#
# license:  Public Domain
#


import struct

objectType  = 0
objectTypes = {}

messageType     = 0
messageTypes    = {}

def pack(fmt, *data):
    return ''.join([struct.pack('!' + i, n) for i, n in zip(fmt, data)])

def unpack(fmt, data):
    n = 0
    values = []
    for i in fmt:
        r = struct.calcsize('!' + i)
        values.append(struct.unpack('!' + data[n:n+r])[0])
        n += r
    return values
        
        
class Parser(type):
    def __init__(klass, name, bases, dict):
        global objectType
        klass.attributes = []
        for k, v in dict.items():
            if not k.startswith('_') and isinstance(v, str):
                klass.attributes.append(k)
        klass.attributes.sort()
        
        klass.fmt     = [dict[i] for i in klass.attributes]
        klass.size    = sum([struct.calcsize(i) for i in klass.fmt])+ 1
        klass.type    = chr(objectType+1) 
        objectTypes[klass.type] = klass
        objectType += 1
        super(Parser, klass).__init__(name, bases, dict)

class MessageTyper(type):
    def __init__(klass, name, bases, dict):
        global messageType
        klass._type    = chr(messageType+1) 
        messageTypes[klass._type] = klass
        messageType += 1
        super(MessageTyper, klass).__init__(name, bases, dict)


class NetObject(object):
    __metaclass__ = Parser
    
    def __init__(self, data=None):
        if not data:
            for k in self.attributes:
                setattr(self, k, None)
        if isinstance(data, str):
            if not data[0] == self.type:
                raise Exception('message type does not match')
            for k, v in zip(self.attributes, struct.unpack(self.fmt, data[1:])):
                setattr(self, '_' + k, v)
                setattr(self, k, v)
            self.read()
        else:
            for k in self.attributes:
                try:                        setattr(self, k, getattr(data, k))
                except AttributeError:      setattr(self, k, None)
            self.prewrite(data)
            for k in self.attributes:
                setattr(self, '_' + k, getattr(self, k))
            self.write()
            
    def data(self):
        print self.fmt
        print [getattr(self, '_' + i) for i in self.attributes]
        return self.type + struct.pack(self.fmt, *[getattr(self, '_' + i) for i in self.attributes])
        
    def prewrite(self, data):
        pass
        
    def write(self):
        pass
        
    def read(self):
        pass
        
    def __repr__(self):
        return '<' + type(self).__name__ + ' ' + ' '.join(['%s=%s' % (k, repr(getattr(self, k))) for k in self.attributes]) + '>'

class NetList(NetObject):
    value = 'c'
    def __init__(self, data=None):
        self.children = []
        if isinstance(data, str):
            if not data[0] == self.type:
                raise Exception('message type does not match')
            length = struct.unpack('!H', data[1:3])
            data = data[3:]
            if not len(data) == self.size * length:
                raise Exception('data is incorrect size')
            self.children = struct.unpack('!%s%s' % (length, self.value), data)
            self.read()
        else:
            self.children = data
            self.prewrite(data)
            self.write()
            
    def data(self):
        return self.type + struct.pack('!H%s%s' % (len(self.children), self.value), *([len(self.children)] + self.children))
        
    def getsize(data):
        length = struct.unpack('!H', data[1:3])
        return length * self.size + 3
    getsize = classmethod(getsize)


class Message(object):
    __metaclass__ = MessageTyper
    def __init__(self, data=None):
        self.children   = []
        self.length = 0 
        if isinstance(data, str):
            if not data[0] == 'S':
                raise Exception('not a space message')
            self.type   = data[1]
            self.__class__ = messageTypes[self.type]
            self.length = struct.unpack('!H', data[2:4])[0]
            if not self.length == len(data):
                raise Exception('data is incorrect size')
            data = data[4:]
            while data:
                type    = objectTypes[data[0]]
                if isinstance(type, NetList):
                    length = NetList.getsize(data)
                else:
                    length = type.size
                if not len(data) >= length:
                    raise Exception('child is incorrect size')
                self.children.append(type(data[0:length]))
                data = data[length:]
        else:
            self.length = 0
            self.type   = self._type
            if data:
                for i in data:
                    self.children.append(i)
        
    def add(self, child):
        self.children.append(child)
        
    def data(self):
        s = ''.join([i.data() for i in self.children])
        self.length = len(s) + 4
        return ''.join(['S', self.type, struct.pack('!H', self.length), s])
        
    def __iter__(self):
        return iter(self.children)        
        
    def __repr__(self):
        return '<%s children=%s length=%s>' % (type(self).__name__, len(self.children), self.length)



