#
# file:     parser2.py
#
# about:    parsing and unparsing of game messages
#
# author:   Tom "Knio" Flanagan
#
# contact:  theknio@gmail.com
#           irc.freenode.net - Knio - #python #pygame #py
#           * comments welcome *
#
# license:  Public Domain
#

protocolVersion = chr(0) # this is version 0

messageType     = -1
messageTypes    = {}

objectType  = -1
objectTypes = {}


import struct

class ParserError(Exception): pass
class ProtocolError(Exception): pass


class Data(object):
    def __init__(self):
        pass
        
    def toData(self, obj):
        pass
        
    def fromData(self, obj):
        pass
        
    def length(self, obj):
        pass
        
class Number(Data):
    def __init__(self, fmt):
        self.fmt = '!' + fmt
    
    def toData(self, obj):
        return struct.pack(self.fmt, obj)
    
    def fromData(self, obj):
        return struct.unpack(self.fmt, obj)[0]
        
    def length(self, object):
        return struct.calcsize(self.fmt)

class String(Data):
    def toData(self, data):
        return chr(len(data)) + data
    
    def fromData(self, data):
        return data[1:1+ord(data[0])]
        
    def length(self, data):
        return 1+ord(data[0])

string  = String()
int8    = Number('b')
uint8   = Number('B')
uint16  = Number('H')
uint32  = Number('I')
float4  = Number('f')
float8  = Number('d')

class MessageType(type):
    def __init__(klass, name, bases, dict):
        global messageType
        klass.type    = chr(messageType+1) 
        messageTypes[klass.type] = klass
        messageType += 1
        super(MessageType, klass).__init__(name, bases, dict)

class ObjectType(type):
    def __init__(klass, name, bases, dict):
        klass.attributes = []
        for k, v in dict.items():
            if not k.startswith('_') and isinstance(v, Data):
                klass.attributes.append((k, v))
        klass.attributes.sort()
    
        global objectType
        klass.type    = chr(objectType+1) 
        objectTypes[klass.type] = klass
        objectType += 1
        super(ObjectType, klass).__init__(name, bases, dict)


class Message(object):
    '''
    Message()
    A whole message. Fits in a UDP packet, sent over the network.
    Contains NetObjects.
    '''
    __metaclass__ = MessageType
    def __init__(self, data=None):
        self.children = []
        if isinstance(data, str):
            self.fromData(data)
        elif isinstance(data, list):
            self.children = data
        elif data != None:
            raise TypeError
        
    def add(self, child):
        self.children.append(child)
        
    append = add
        
    def toData(self):
        return 'S' + protocolVersion + self.type + ''.join([i.toData() for i in self.children])
    
    def fromData(self, data):
        if not data[0] == 'S':              raise ParserError('packet does not start with "S"')
        if not data[1] == protocolVersion:  raise ParserError('wrong protocol version. consider upgrading your client')
        self.children  = []
        self.__class__ = messageTypes[data[2]]
        n = 3
        while n < len(data):
            obj = NetObject()
            l = obj.fromData(data[n:])
            self.children.append(obj)
            n += l
        
    #TODO: consider making this a subclass of list
    def __getitem__(self, item):
        return self.children[item]
        
    def __iter__(self):
        return iter(self.children)
        
    def __len__(self):
        return len(self.children)
        
    def __repr__(self):
        return '<%s children=%s>' % (type(self).__name__, len(self.children))



class NetObject(object):
    '''
    NetObject()
    a collection of Data, representing a single game object.
    '''
    __metaclass__ = ObjectType
    def __init__(self, data=None, game=None, **kwargs):
        if isinstance(data, str):
            self.fromData(data)
        elif data and game:
            self.fromObject(data, game)
        elif hasattr(data, 'game'):
            self.fromObject(data, data.game)
        elif data:
            raise TypeError('NetObject: don\'t know how to handle that data object')
        self.__dict__.update(kwargs)

    def toData(self):
        s = []
        for k, v in self.attributes:
            d = getattr(self, k)
            if d is v:  raise AttributeError('%s instance does not have a \'%s\' attribute set' % (type(self), k))
            s.append(v.toData(d))
        return self.type + ''.join(s)
            
    
    def fromData(self, data):
        self.__class__ = objectTypes[data[0]] # this needs to be done first, so that we have the correct self.attributes
        n = 1
        for k, v in self.attributes:
            l = v.length(data[n:])
            setattr(self, k, v.fromData(data[n:n+l]))
            n += l
        return n
            
        
    def toObject(self, object, game):
        raise NotImplementedError
        
    def fromObject(self, object, game):
        raise NotImplementedError   

    def __repr__(self):
        return '<' + type(self).__name__ + ' ' + ' '.join(['%s=%s' % (k, type(getattr(self, k)) is float and '%.2f' % getattr(self, k) or repr(getattr(self, k))) for k,v in self.attributes]) + '>'



#
# Unit Test!
#


if __name__ == '__main__':
    
    # define some objects..
    
    class MooMessage(Message):  pass
    class Moo(NetObject):
        foo = uint32
        bar = string
    
    # create some instances
    
    msg = MooMessage()
    
    moo = Moo()
    moo.foo = 0xFFFFFFFF
    moo.bar = 'ABCDEFG'
    
    msg.add(moo)
    
    # test it!
    data = msg.toData()
    print 'MooMessage', repr(data)
    
    msg2 = Message(msg.toData())
    assert isinstance(msg2, MooMessage)
    assert len(msg2.children) == 1
    moo2 = msg2.children[0]
    assert moo2.foo == moo.foo
    assert moo2.bar == moo.bar
    
    assert msg.toData() == msg2.toData()
    
    
    # it worked!!
    
    
    
    
    
    
    
    
    
