#
# file:     player.py
#
# about:    player class
#
# author:   Tom "Knio" Flanagan
#
# contact:  theknio@gmail.com
#           irc.freenode.net - Knio - #python #pygame #py
#           * comments welcome *
#
# license:  Public Domain
#

from vector import Vector, PI, CI
from object import Object
from laser  import Laser
from planet import Planet
import math


class Player(Object):
    thrustrate      = 0.5
    turnrate        = 1.0
    
    def __init__(self, game, user=None, pos=(0., 0., 0., 0., 0.), id=None):
        Object.__init__(self, game, pos, id=id)
        self.user       = user
        if self.user:
            self.user.player= self
        self.thrusting  = 0. #NOTE: make these 0-255 (byte) values?
        self.turning    = 0. #
        self.firing     = 0
        self.fuel       = 1. #
        self.hp         = 1. #
        self.reload     = 0
        self.orbit      = []
        self.orbitvect  = Vector(self)
        
    def init(self):
        self.game.addPlayer(self)
        
    def tick(self):
        Object.tick(self)
        self.gravity()
        self.collide()
        self.thrust()
        self.turn()
        self.fire()
        self.reload = max(self.reload-1, 0)
        if self.orbit:
            self.orbit.pop(0)
        
    def hit(self, what):
        if isinstance(what, Planet):
            self.orbit = []
            self.orvitvect = Vector(self)
            self.collision(what)
            if not self.thrusting:
                self.mul(0.5)
            if (not self.thrusting) and self.m < 0.8:
                self.m = 0.0
                what.dock(self)
        if isinstance(what, Laser):
            self.hp -= 1/8.
            if self.hp <= 0.:
                self.die()
    
    def die(self):
        self.user.die()
        self.game.removePlayer(self)
     
    def thrust(self):
        if not self.fuel > 0.:
            self.thrusting = 0.
            return
        self.add(Vector(0, 0, self.f, self.thrustrate * self.thrusting / 4.))
        self.fuel = max(self.fuel - abs(self.thrusting) / 1024., 0.)
        if self.thrusting != 0.0:
            self.orbit = []
            self.orbitvect = Vector(self)
        
    def turn(self):
        self.f += self.turnrate * self.turning / 6.0
        
    def fire(self):
        if self.firing and self.reload == 0:
            vect = Vector(self)
            vect.add(self.f, 40.0)
            Laser(self.game, self, (vect.x, vect.y, vect.a, vect.m, self.f))
            self.reload = 10
        
    def collision(self, planet):
        v = Vector(planet)
        a = planet.angle(self)
        v.move(a, planet.size) # move to the surface
        self.x = v.x
        self.y = v.y
                
        v = Vector()
        v.a = a
        f = math.cos(a - (self.a + PI))
        v.m = f * self.m # apply opposite vector
        self.add(v)
        if v.m > 10.:
            #print v.m
            self.hp -= (v.m-10) / 40.
            if self.hp <= 0.:
                self.die()
        self.orbit = []
        self.orbitvect = Vector(self)

    def calcOrbit(self):
        v = self.orbitvect
        g = self.game
        o = self.orbit
        for i in range(min(100, 5000-len(self.orbit))):
            if not i % 10:
                for i in g.planets.values():
                    if i.distance(v) <= i.size:
                        return
            v.move()
            v.add(g.getGravity(v.x, v.y))
            o.append((v.x, v.y))

    def __setattribute__(self, name, value):
        if name == 'user':
            value.player = self
        return object.__setattribute__(self, name, value)

    def __repr__(self):
        return '<Player(id=%s, user=%s, %.2f, %.2f, %.2f, %.2f>' % (self.id, self.user.id, self.x, self.y, self.a, self.m)






