#
# file:     vector.py
#
# about:    simple 2d point/vector class
#
# author:   Tom "Knio" Flanagan
#
# contact:  theknio@gmail.com
#           irc.freenode.net - Knio - #python #pygame #py
#           * comments welcome *
#
# license:  Public Domain
#

__author__      = 'Tom "Knio" Flanagan'
__version__     = 1.1
__license__     = 'Public Domain'
__copyright__   = None

import math

PI = math.pi
RI = PI/2
CI = 2*PI

NORTH   = RI
EAST    = 0
SOUTH   = -RI
WEST    = PI

class Vector(object):
    '''
    A Point/Vector
    '''
    __slots__ = ['x', 'y', 'a', 'm']
    def __init__(self, x=0., y=0., a=0., m=0.):
        '''
        Vector(x=0., y=0., a=0., m=0.)
        Vector(Vector) # make a copy
        
        Create a new Vector
        '''
        if isinstance(x, Vector): # got vector as first argument, make a copy
            return self.__init__(x.x, x.y, x.a, x.m)
        if x is None:
            return self.__init__()
        self.x = float(x)
        self.y = float(y)
        self.a = float(a)
        self.m = float(m)

    def __repr__(self):
        return '<Vector(%s, %s, %s, %s)>' % (self.x, self.y, self.a, self.m)
        
    
    def angle(self, obj):
        '''
        Vector.angle(obj)
        
        Returns the angle from here to obj.
        '''
        if obj is None:
            obj = Vector()
        if isinstance(obj, Vector):
            x = self.x - obj.x 
            y = self.y - obj.y
        return -math.atan2(y,-x)

    def direction(self, a):
        if isinstance(a, Vector):
            a = self.angle(a)
        d = (a - self.a) % CI
        if d >  PI: d -= CI
        if d < -PI: d += CI
        return d

    def distance(self, x, y=None):
        '''
        Vector.distance(obj)
        
        Returns the distance from self to obj.
        '''
        if x is None:
            x = Vector()
        if isinstance(x, Vector):
            y = self.y - x.y
            x = self.x - x.x
        return math.hypot(x,y)

    def move(self, a=None, m=None):
        '''
        Vector.move(angle, distance)
        Vector.move(Vector)
        
        Moves the Vector distance far away at an angle of angle.
        if no distance/angle provided, moves by own vector angle/magnitude
        '''
        if a is None:
            a = self
        if isinstance(a, Vector):
            m = a.m
            a = a.a
        self.x += math.cos(a) * m
        self.y += math.sin(a) * m

    def add(self, a, m=None):
        '''
        Vector.add(vector)
        
        Add a vector.
        '''
        #x, y, a, m = 
        vect = Vector(self)
        vect.move()
        vect.move(a, m)
        
        self.a = self.angle(vect)
        self.m = self.distance(vect)
        


    def sub(self, vector):
        '''
        Vector.add(vector)
        
        Same as Vector.add(), but reverses the direction.
        '''
        vect = Vector(vector)
        vect.a += PI
        self.add(vect)


    def div(self, x):
        '''
        Vector.div(x)
        
        Divide a Vector
        '''
        self.m /= x

    def mul(self, x):
        '''
        Vector.mul(x)
        
        Multiply a Vector
        '''
        self.m *= x

