#
# starfield.py -- a scrolling starfield
# 
# author: Tom "Knio" Flanagan
#         knio@h4xn3t.net
#         theknio@gmail.com
#         irc.freenode.net - Knio - #python #pygame #py
#
#         comments welcome!
#
# this file released under Public Domain
#

__author__      = "Tom Flanagan"
__copyright__   = "Public Domain"
__version__     = 1.0
__license__     = "Public Domain"

import pygame
import random
from space.vector import *
from pygame.locals import *

NUMSTARS = 400
SCREENSIZE = (640,640)


class Starfield(object):
    def __init__(self, size, stars):
        self.size = size
        self.surf = pygame.Surface(size)
        self.fade = pygame.Surface(size)
        self.fade.set_alpha(128)
        self.stars = []
        self.vel = Vector()
        for i in xrange(stars):
            x = random.random()*size[0]
            y = random.random()*size[1]
            depth = random.random()
            self.stars.append(Star(Vector(x, y), depth, size))
        self.move()

    def move(self, vec=None):
        vec = vec or self.vel
        a = vec.a + PI
        m = vec.m
        for i in self.stars:
            i.move(a, m)
        
    def render(self, surf=None):
        self.surf.blit(self.fade, (0,0))    # alpha blend - a bit slower, makes a motion blur
        #self.surf.fill((0,0,0))             # fill - a bit faster, no motion blur
        #self.surf = pygame.Surface(self.size)
        for i in self.stars:
            i.render(self.surf)
        if surf:
            surf.blit(self.surf, (0,0))



class Star(object):
    def __init__(self, pos, depth, size):
        self.pos    = pos   
        self.oldpos = pos
        self.depth  = depth
        self.size   = size

    def move(self, a, m):
        self.oldpos = (self.pos.x, self.pos.y)
        self.pos.move(a, m*self.depth)
        self.pos.x %= self.size[0]
        self.pos.y %= self.size[1]
        
        
    def render(self, surf):
        if abs(self.pos.x-self.oldpos[0]) > self.size[0]/2 or abs(self.pos.y-self.oldpos[1]) > self.size[1]/2:
            return # star has wrapped, dont want to draw a streak
        pygame.draw.line(surf, (self.depth*255,)*3, (self.pos.x, self.size[1]-self.pos.y), (self.oldpos[0], self.size[1]-self.oldpos[1]), 1)
        #surf.fill((self.depth*255,)*3, (self.pos.x, self.pos.y, 1, 1))
        
def main():
    pygame.init()
    screen = pygame.display.set_mode(SCREENSIZE)
    stars = Starfield(SCREENSIZE, NUMSTARS)
    stars.vel.add(Vector(0, 0, 1, 3))
    done = False
    clock = pygame.time.Clock()

    while not done:
        clock.tick()
        screen.fill((0,0,0))
        stars.move()
        stars.render(screen)
        pygame.display.flip()

        print '\b'*100, 'FPS:', clock.get_fps(),

        for i in pygame.event.get():
            if i.type == QUIT:
                done = True
            if i.type == KEYDOWN:
                if i.key == K_ESCAPE:   done = True
                if i.key == K_UP:       stars.vel.add(Vector(0, 0, NORTH, 1))
                if i.key == K_DOWN:     stars.vel.add(Vector(0, 0, SOUTH, 1))
                if i.key == K_LEFT:     stars.vel.add(Vector(0, 0, WEST,  1))
                if i.key == K_RIGHT:    stars.vel.add(Vector(0, 0, EAST,  1))

    pygame.quit()

if __name__ == '__main__':
    main()
        

        
