#
# file:     space.py
#
# about:    simple viewer
#
# author:   Tom "Knio" Flanagan
#
# contact:  knio@h4xn3t.net
#           theknio@gmail.com
#           irc.freenode.net - Knio - #python #pygame #py
#           * comments welcome *
#
# license:  Public Domain
#


# TODO: zooming. faster = zoom out, + some keys
# TODO: camera following


import pygame
import random
import os
from object     import Object
from planet     import Planet
from player     import Player
from laser      import Laser
from minimap    import Minimap
from starfield      import Starfield
from space.vector   import Vector, PI, CI
pygame.init()

class View(object):
    def __init__(self, game, size, user=None):
        self.init() 
        self.turn       = 0
        self.game       = game
        self.w, self.h  = size
        self.user       = user
        self.camerazoom = 1.0
        self.userzoom   = 1.0
        self.zoom       = 1.0
        pygame.display.set_icon(pygame.image.load(Object.image)) # to not crash in py2exe
        self.screen = pygame.display.set_mode(size, pygame.DOUBLEBUF)
        self.stars  = Starfield(size, (self.w * self.h)/4096)
        self.camera = Vector(self.user and self.user.player or None)
        self.font   = pygame.font.Font('fonts/verdana.ttf', 10)
        self.map    = Minimap(self, game, 300, 60)

    def init(self):
        pygame.init()

    def setUser(self, user):
        self.user   = user
        self.camera = Vector(user.player)
        
    def pos(self, (x, y)):
        x = int(( x - self.camera.x) / self.zoom + self.h/2 )
        y = int((-y + self.camera.y) / self.zoom + self.w/2 )
        return x, y
        
    def tick(self, t):
        self.move()
        if t < 0:
            return
        self.turn += 1
        self.render()
        self.renderOrbit()
        self.renderLine()
        for i in self.game.objects.values():
            self.renderObject(i)
        for i in self.game.planets.values():
            self.renderPlanet(i)
        for i in self.game.players.values():
            self.renderPlayer(i)
        self.renderInfo()
        self.map.tick()
    
    def move(self):
        # this function is ugly..
        if not self.user.player:
            self.camera = Vector()
            self.zoom = 20
            cam = self.camera
            cam.div(self.zoom)
            self.stars.move(cam)
            return
        self.camera.move()
        d = self.camera.distance(self.user.player)
        a = self.camera.angle(self.user.player) % CI
        cam = Vector(self.camera)
        obj = Vector(self.user.player)
        obj.a += PI
        cam.add(obj) # cam is relative to obj
        cam.div(4)
        cam.a += PI
        cam.add(a, d/512)
        self.camera.add(cam)
        zoom = self.camera.m/8 + 1
        self.camerazoom += (zoom-self.camerazoom)/8
        self.zoom = self.camerazoom * self.userzoom
        
        cam = Vector(self.camera)
        cam.div(self.camerazoom)
        
        
    def render(self):
        pygame.display.flip()
        self.screen.fill((0,0,0))
        self.stars.render(self.screen)
        
    def renderOrbit(self):
        if not self.user.player:
            return
        self.user.player.calcOrbit() # TODO: why is this being called here?
        lines = map(self.pos, self.user.player.orbit[::int(self.zoom)+1])
        if len(lines) > 1:
            pygame.draw.lines(self.screen, self.user.team.color, False, lines, 1)
            
    def renderLine(self):
        if not self.user.player:
            return
        v = Vector(self.user.player)
        start   = self.pos((v.x, v.y))
        v.move(self.user.player.f, 5000)
        end     = self.pos((v.x, v.y))
        pygame.draw.line(self.screen, [i/2 for i in self.user.team.color], start, end, 1)
        
    def renderInfo(self):
        if not self.user.player:
            return
        self.screen.blit(self.font.render("Fuel:", 1, (255,255,255)), (10,10))
        self.screen.fill(self.user.team.color, (50, 10, (self.w-50)*self.user.player.fuel, 10))
        self.screen.blit(self.font.render("Health:", 1, (255,255,255)), (10,30))
        self.screen.fill(self.user.team.color, (50, 30, (self.w-50)*self.user.player.hp, 10))
    
    
    def renderObject(self, i):
        if not hasattr(i, 'view'):
            if type(i).__name__ == 'Laser':
                i.view = Laser(i, self)
            else:
                i.view = Object(i, self)
        i.view.tick()
            
    def renderPlanet(self, i):
        if not hasattr(i, 'view'):
            i.view = Planet(i, self)
        i.view.tick()
            
    def renderPlayer(self, i):
        if not hasattr(i, 'view'):
            i.view = Player(i, self)
        i.view.tick()
    
    def quit(self):
        pygame.quit()
        